<?php
require_once 'config.php';

$conn = connectDB();
$method = $_SERVER['REQUEST_METHOD'];

switch ($method) {
    case 'GET':
        // Get all users or a specific user
        $id = isset($_GET['id']) ? (int)$_GET['id'] : null;
        
        if ($id) {
            $stmt = $conn->prepare("SELECT id, username, name, role, created_at FROM users WHERE id = ?");
            $stmt->bind_param("i", $id);
            $stmt->execute();
            $result = $stmt->get_result();
            $user = $result->fetch_assoc();
            
            if (!$user) {
                returnError("User not found", 404);
            }
            
            returnResponse($user);
        } else {
            $stmt = $conn->prepare("SELECT id, username, name, role, created_at FROM users ORDER BY name");
            $stmt->execute();
            $result = $stmt->get_result();
            $users = $result->fetch_all(MYSQLI_ASSOC);
            
            returnResponse($users);
        }
        break;
        
    case 'POST':
        if (isset($_GET['login'])) {
            // Login
            $data = getRequestBody();
            
            if (!isset($data['username']) || !isset($data['password'])) {
                returnError("Username and password are required", 400);
            }
            
            $stmt = $conn->prepare("SELECT id, username, name, role, password FROM users WHERE username = ?");
            $stmt->bind_param("s", $data['username']);
            $stmt->execute();
            $result = $stmt->get_result();
            $user = $result->fetch_assoc();
            
            if (!$user || !password_verify($data['password'], $user['password'])) {
                returnError("Invalid username or password", 401);
            }
            
            // Log the login activity
            $stmt = $conn->prepare("INSERT INTO user_logs (user_id, action, details) VALUES (?, 'Đăng nhập', 'Đăng nhập thành công')");
            $stmt->bind_param("i", $user['id']);
            $stmt->execute();
            
            // Remove password from response
            unset($user['password']);
            
            returnResponse($user);
        } else {
            // Add a new user
            $data = getRequestBody();
            
            // Validate required fields
            if (!isset($data['username']) || !isset($data['password']) || !isset($data['name'])) {
                returnError("Username, password, and name are required", 400);
            }
            
            // Check if username already exists
            $stmt = $conn->prepare("SELECT id FROM users WHERE username = ?");
            $stmt->bind_param("s", $data['username']);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows > 0) {
                returnError("Username already exists", 400);
            }
            
            // Create user
            $stmt = $conn->prepare("INSERT INTO users (username, password, name, role) VALUES (?, ?, ?, ?)");
            $hashedPassword = password_hash($data['password'], PASSWORD_DEFAULT);
            $role = isset($data['role']) ? $data['role'] : 'user';
            
            $stmt->bind_param("ssss", 
                $data['username'], 
                $hashedPassword, 
                $data['name'], 
                $role
            );
            
            if (!$stmt->execute()) {
                returnError("Failed to create user: " . $conn->error);
            }
            
            $id = $conn->insert_id;
            
            // Return the new user without password
            $stmt = $conn->prepare("SELECT id, username, name, role, created_at FROM users WHERE id = ?");
            $stmt->bind_param("i", $id);
            $stmt->execute();
            $result = $stmt->get_result();
            $user = $result->fetch_assoc();
            
            returnResponse($user, 201);
        }
        break;
        
    case 'PUT':
        // Update a user
        $id = isset($_GET['id']) ? (int)$_GET['id'] : null;
        
        if (!$id) {
            returnError("User ID is required", 400);
        }
        
        $data = getRequestBody();
        
        // Check if user exists
        $stmt = $conn->prepare("SELECT id FROM users WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 0) {
            returnError("User not found", 404);
        }
        
        // Build update query
        $sql = "UPDATE users SET";
        $params = [];
        $types = "";
        $updates = [];
        
        if (isset($data['name'])) {
            $updates[] = " name = ?";
            $params[] = $data['name'];
            $types .= "s";
        }
        
        if (isset($data['role'])) {
            $updates[] = " role = ?";
            $params[] = $data['role'];
            $types .= "s";
        }
        
        if (isset($data['password']) && !empty($data['password'])) {
            $updates[] = " password = ?";
            $params[] = password_hash($data['password'], PASSWORD_DEFAULT);
            $types .= "s";
        }
        
        if (empty($updates)) {
            returnError("No fields to update", 400);
        }
        
        $sql .= implode(",", $updates);
        $sql .= " WHERE id = ?";
        $params[] = $id;
        $types .= "i";
        
        $stmt = $conn->prepare($sql);
        $stmt->bind_param($types, ...$params);
        
        if (!$stmt->execute()) {
            returnError("Failed to update user: " . $conn->error);
        }
        
        // Return the updated user
        $stmt = $conn->prepare("SELECT id, username, name, role, created_at FROM users WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $result = $stmt->get_result();
        $user = $result->fetch_assoc();
        
        returnResponse($user);
        break;
        
    case 'DELETE':
        // Delete a user
        $id = isset($_GET['id']) ? (int)$_GET['id'] : null;
        
        if (!$id) {
            returnError("User ID is required", 400);
        }
        
        // Check if user exists
        $stmt = $conn->prepare("SELECT id FROM users WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 0) {
            returnError("User not found", 404);
        }
        
        // Delete user
        $stmt = $conn->prepare("DELETE FROM users WHERE id = ?");
        $stmt->bind_param("i", $id);
        
        if (!$stmt->execute()) {
            returnError("Failed to delete user: " . $conn->error);
        }
        
        returnResponse(['message' => 'User deleted successfully']);
        break;
        
    default:
        returnError('Method not allowed', 405);
}

$conn->close();
?>